import { analyzeURL } from "./engine.js";

const analyzeBtn = document.getElementById("analyzeBtn");
const urlInput = document.getElementById("urlInput");

const welcomeScreen = document.getElementById("welcomeScreen");
const appScreen = document.getElementById("appScreen");
const startBtn = document.getElementById("startGuardian");

const welcomeBackScreen = document.getElementById("welcomeBackScreen");
const continueBtn = document.getElementById("continueGuardian");

// First-time check
const hasVisited = localStorage.getItem("guardianVisited");

// Decide which screen to show
if (hasVisited) {
  welcomeScreen.style.display = "none";
  welcomeBackScreen.classList.add("active");
} else {
  localStorage.setItem("guardianVisited", "true");
}

function enterGuardian() {
  buttonClickSound.currentTime = 0;
  buttonClickSound.play();

  welcomeScreen.classList.remove("active");
  welcomeBackScreen.classList.remove("active");

  setTimeout(() => {
    welcomeScreen.style.display = "none";
    welcomeBackScreen.style.display = "none";
    appScreen.classList.add("active");
  }, 300);
}

startBtn.addEventListener("click", enterGuardian);
continueBtn.addEventListener("click", enterGuardian);


// --- Sound Effects ---
const buttonClickSound = new Audio("buttonclick.wav");
const safeOrWarnSound = new Audio("safeorwarn.mp3");
const dangerSound = new Audio("danger.wav");

// Volume tuning
buttonClickSound.volume = 0.35;
safeOrWarnSound.volume = 0.45;
dangerSound.volume = 0.55;


// START GUARDIAN button
startBtn.addEventListener("click", () => {
  buttonClickSound.currentTime = 0;
  buttonClickSound.play();

  welcomeScreen.classList.remove("active");

  setTimeout(() => {
    welcomeScreen.style.display = "none";
    appScreen.classList.add("active");
  }, 300);
});


function generateHumanSummary(result) {
  // SAFE CASE
  if (result.verdict === "SAFE") {
    return "This link appears safe. No suspicious patterns, deceptive wording, or risky domain behavior were detected. The structure and wording of the URL match what is typically seen on legitimate websites.";
  }

  const keywords = [];
  let hasRiskyTLD = false;
  let hasEntropy = false;
  let hasHomoglyph = false;

  result.reasons.forEach(r => {
    if (r.includes("keyword")) {
      const word = r.match(/\"(.+?)\"/)?.[1];
      if (word) keywords.push(word);
    }
    if (r.includes("TLD")) hasRiskyTLD = true;
    if (r.includes("entropy")) hasEntropy = true;
    if (r.includes("homoglyph")) hasHomoglyph = true;
  });

  let intro =
    result.verdict === "WARNING"
      ? "This link shows some patterns commonly associated with suspicious websites. "
      : "This website displays several strong indicators commonly linked to phishing or fraud. ";

  let text = intro;

  if (keywords.length) {
    text += `It uses words like ${keywords.join(", ")}, which are often used to pressure users into taking urgent action. `;
  }

  if (hasRiskyTLD) {
    text += "It operates on a domain extension that is frequently abused for malicious activity. ";
  }

  if (hasEntropy) {
    text += "The domain structure appears irregular, suggesting it may have been automatically generated. ";
  }

  if (hasHomoglyph) {
    text += "There are signs of deliberate character substitutions, a known technique used to imitate trusted brands. ";
  }

  text +=
    result.verdict === "WARNING"
      ? "While this does not confirm malicious intent, caution is advised."
      : "Taken together, these patterns indicate the link should not be trusted.";

  return text;
}


function typeText(element, text, speed = 18) {
  element.textContent = "";
  let i = 0;

  const interval = setInterval(() => {
    element.textContent += text[i];
    i++;
    if (i >= text.length) clearInterval(interval);
  }, speed);
}


// ANALYZE button
analyzeBtn.addEventListener("click", () => {
  buttonClickSound.currentTime = 0;
  buttonClickSound.play();

  const url = urlInput.value.trim();
  if (!url) return;

  const result = analyzeURL(url);

  // Result-based sound
  if (result.verdict === "DANGEROUS") {
    dangerSound.currentTime = 0;
    dangerSound.play();
  } else {
    safeOrWarnSound.currentTime = 0;
    safeOrWarnSound.play();
  }

  const resultBox = document.getElementById("result");
  resultBox.classList.remove("show");
  void resultBox.offsetWidth;
  resultBox.classList.add("show");

  const verdictEl = document.getElementById("verdict");
  const scoreFill = document.getElementById("scoreFill");
  const explanationEl = document.getElementById("explanation");

  verdictEl.textContent = `${result.verdict} (${result.score})`;

  scoreFill.style.width = Math.min(result.score, 100) + "%";
  scoreFill.style.background =
    result.verdict === "SAFE" ? "#2ecc71" :
    result.verdict === "WARNING" ? "#f1c40f" :
    "#e74c3c";

  const humanSummary = generateHumanSummary(result);
  typeText(explanationEl, humanSummary);
});


const guardianSite = document.getElementById("guardianSite");

if (guardianSite) {
  guardianSite.addEventListener("click", (e) => {
    e.preventDefault();
    chrome.tabs.create({
      url: "https://guardian-v1.netlify.app"
    });
  });
}
